*-------------------------------------------------------------------------------
/* title : Compute intra- and inter-industry proxies split based on the level of
		   development for each country
   source: World Input-Output Tables, 2016 Release (in Stata format)
		   http://www.wiod.org/database/wiots16	 
   input : $data/wiot_stata_Nov16/WIOT`year'_October16_ROW.dta
   output: $temp/Proxies_development_split.dta */
*-------------------------------------------------------------------------------

*===============================================================================
* Set up
*===============================================================================
*--------------------------SET-UP & READ DIRECTORIES----------------------------
cd												// Current directory of code
do "[0.0] directories.do"

global developed  AUS AUT BEL CAN CHE DEU DNK FIN FRA GBR JPN KOR NLD NOR SWE TWN USA
global developing ROW BGR BRA CHN CZE CYP ESP EST GRC HRV HUN IDN IND IRL ITA LTU LVA LUX MEX MLT POL PRT ROU RUS SVK SVN TUR
*===============================================================================
* Intra-industry proxies
*===============================================================================

* Create intra-industry importing shares (IM)
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		keep country year IndustryCode IndustryDescription id *`country'*
		gen partner=""
		foreach p of global developed {
			replace partner="dvlpd" if country=="`p'"
		}
		foreach p of global developing {
			replace partner="dvlpng" if country=="`p'"
		}
		replace partner="`country'" if country=="`country'"

		gcollapse (sum) v`country'1-v`country'56, by(year partner) 
		foreach j of varlist v`country'1-v`country'56 {
			gegen `j'_sum = total(`j'), by(year)
			gen IM_`j' = `j'/(`j'_sum)
			replace IM_`j' = 0 if `j'==0 & `j'_sum==0
			drop `j' `j'_sum
		}
		rename IM_v`country'* IM_* 
		drop if partner=="`country'"
		greshape long IM_, i(year partner) j(industry) string
		greshape wide IM_, i(year industry) j(partner) string
		cap gen country="`country'"
		destring industry, replace force
		keep country year industry IM*
		sort industry year	
		save "$temp/IM`country'_`y'_development_split.dta", replace
	}
	noi dis "`country'"
}

* Create intra-industry exporting shares (EX)
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		keep if country=="`country'"
		forvalues j=1/56 {	
			gen v`j'_ttl = 0 
			foreach c in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN CHN AUS BRA CAN CHE CYP IDN IND JPN KOR LUX MEX MLT RUS TUR TWN USA ROW {
				replace v`j'_ttl = v`j'_ttl + v`c'`j' 
			}
			gen v`j'_dvlpd = 0
			gen v`j'_dvlpng = 0
			foreach p of global developed {
				replace v`j'_dvlpd = v`j'_dvlpd + v`p'`j' 
				replace v`j'_dvlpd = v`j'_dvlpd- v`p'`j' if country=="`country'" & country=="`p'"
			}
			foreach p of global developing {
				replace v`j'_dvlpng = v`j'_dvlpng + v`p'`j'
				replace v`j'_dvlpng = v`j'_dvlpng - v`p'`j' if country=="`country'" & country=="`p'"
			}
		}
		gen ttl = 0
		gen ttl_dvlpd = 0
		gen ttl_dvlpng = 0
		forvalues j=1/56 {	
			replace ttl = ttl + v`j'_ttl
			replace ttl_dvlpd = ttl_dvlpd + v`j'_dvlpd
			replace ttl_dvlpng = ttl_dvlpng + v`j'_dvlpng
		}
		gen EX_dvlpd = ttl_dvlpd/ttl
		gen EX_dvlpng = ttl_dvlpng/ttl
		foreach j in _dvlpd _dvlpng {
			replace EX`j' = 0 if EX`j'==.
		}
		keep country year id EX*
		rename id industry
		sort industry year
		save "$temp/EX`country'_`y'_development_split.dta", replace
	}
	noi dis "`country'"
}

*===============================================================================
* Inter-industry proxies that include all supply links (broad)
*===============================================================================

* STEP 1: construct "matrices" of weights from domestic IO tables
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		keep if Country=="`country'"
		rename Country country
		rename RNr id
		drop if id>56
		drop *57 *58 *59 *60 *61
		keep country id *`country'*
		rename v`country'* i*

		gegen i_rowsum=rowtotal(i1-i56)
		forvalues v=1/56 {
			gen c_down`v' = i`v'/i_rowsum
			replace c_down`v'=0 if c_down`v'==. 	// need zeros for matrix manipulation later on
		}
		forvalues v=1/56 {
			gegen i`v'_colsum=total(i`v')
			gen c_up`v' = i`v'/i`v'_colsum
			replace c_up`v'=0 if c_up`v'==. 		// need zeros for matrix manipulation later on
			drop i`v'_colsum
		}
		forvalues v=1/56 {
			gen c_down`v'_0D = c_down`v' 
			replace c_down`v'_0D = 0 if id==`v' 
		}
		forvalues v=1/56 {
			gen c_up`v'_0D = c_up`v' 
			replace c_up`v'_0D = 0 if id==`v' 
		}
		rename id bid
		drop i_rowsum i*
		gen year = `y'
		rename bid industry
		order country year industry
		compress
		save "$temp/COEF`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 2a: proxies of downIM, upEX, upIM, downEX with time-varying weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IM`country'_`y'_development_split.dta", clear
		merge 1:1 country industry year using "$temp/EX`country'_`y'_development_split.dta"
		drop _merge
		merge 1:1 country industry year using "$temp/COEF`country'_`y'.dta"
		drop _merge
		order country industry year 
		
		mkmat IM_dvlpng, matrix(IM_dvlpng)
		mkmat IM_dvlpd, matrix(IM_dvlpd) 
		mkmat EX_dvlpng, matrix(EX_dvlpng)
		mkmat EX_dvlpd, matrix(EX_dvlpd)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in IM_dvlpng IM_dvlpd EX_dvlpng EX_dvlpd {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			rename down0d`type'* down0d`type'
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			rename up0d`type'* up0d`type'
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
			rename down`type'* down`type'
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
			rename up`type'* up`type'
		}
		keep country industry year IM* EX* down* up* 
		foreach var in downIM_dvlpng downIM_dvlpd upEX_dvlpng upEX_dvlpd ///
						down0dIM_dvlpng down0dIM_dvlpd up0dEX_dvlpng up0dEX_dvlpd {
			rename `var' `var'broad
		}
		save "$temp/IM`country'_`y'_TV.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_TV.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_TV.dta"
	}
	sort country industry year
	save "$temp/Proxies_`country'.dta", replace
}

use "$temp/Proxies_AUT.dta", clear
qui foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/Proxies_`country'.dta"
}
sort country industry year
save "$temp/Proxies_development_split.dta", replace

* Housekeeping
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		cap erase "$temp/IM`country'_`y'_development_split.dta"
		cap erase "$temp/IM`country'_`y'_TV.dta"
		cap erase "$temp/IM`country'_`y'_00.dta"
		cap erase "$temp/COEF`country'_`y'.dta"
		cap erase "$temp/EX`country'_`y'_development_split.dta"
		cap erase "$temp/Proxies_`country'_00.dta"
		cap erase "$temp/Proxies_00.dta"
	}
}

*===============================================================================
* Additional inter-industry proxies based on the observed & not all supply links
*===============================================================================

* Create intra-industry importing shares (IM)
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		keep country year IndustryCode IndustryDescription id *`country'*
		gen partner=""
		foreach p of global developed {
			replace partner="dvlpd" if country=="`p'"
		}
		foreach p of global developing {
			replace partner="dvlpng" if country=="`p'"
		}
		replace partner="`country'" if country=="`country'"

		rename id prod
		gcollapse (sum) v`country'1-v`country'56, by(year prod partner) 
		foreach v of varlist v`country'1-v`country'56 { // v is industry that imports "product" id (supplier-industry)
			gegen `v'_sum = total(`v'), by(year prod)
			gen IM_`v' = `v'/(`v'_sum)
			replace IM_`v' = 0 if `v'==0 & `v'_sum==0
			drop `v' `v'_sum
		}
		rename IM_v`country'* IM_* 

		drop if partner=="`country'"
		
		greshape long IM_, i(year prod partner) j(industry) string
		
		tostring prod, gen(partnerprod)
		forvalues v=1/9 {
			replace partnerprod ="0`v'" if partnerprod=="`v'"
			}
		replace partnerprod = partner + partnerprod
		drop partner prod
		greshape wide IM_, i(year industry) j(partnerprod) string

		gen country="`country'"
		destring industry, replace force
		keep country year industry IM*
		order country
		sort industry year	
		save "$temp/IM`country'_`y'_development_split.dta", replace
	}
	noi dis "`country'"
}

* Create intra-industry exporting shares (EX)
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		keep if country=="`country'"
		forvalues v=1/56 {	
			gen v`v'_ttl = 0 
			foreach c in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN CHN AUS BRA CAN CHE CYP IDN IND JPN KOR LUX MEX MLT RUS TUR TWN USA ROW {
				replace v`v'_ttl = v`v'_ttl + v`c'`v' 
			}
		gen v`v'_dvlpd = 0
		gen v`v'_dvlpng = 0
		foreach p of global developed {
			replace v`v'_dvlpd = v`v'_dvlpd + v`p'`v'
			replace v`v'_dvlpd = v`v'_dvlpd- v`p'`v' if country=="`country'" & country=="`p'"
		}
		foreach p of global developing {
			replace v`v'_dvlpng = v`v'_dvlpng + v`p'`v'
			replace v`v'_dvlpng = v`v'_dvlpng - v`p'`v' if country=="`country'" & country=="`p'"
			}
		}
		forvalues v=1/56 {	
			gen EX`v'_dvlpd = v`v'_dvlpd/v`v'_ttl
			replace EX`v'_dvlpd=0 if EX`v'_dvlpd==.
		}
		forvalues v=1/56 {	
			gen EX`v'_dvlpng = v`v'_dvlpng/v`v'_ttl
			replace EX`v'_dvlpng=0 if EX`v'_dvlpng==.
		}
		keep country year id EX*
		rename id industry
		sort industry year
		save "$temp/EX`country'_`y'_development_split.dta", replace
	}
	noi dis "`country'"
}

* STEP 1: construct "matrices" of weights from domestic IO tables
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		keep if Country=="`country'"
		rename Country country
		rename RNr id
		drop if id>56
		drop *57 *58 *59 *60 *61
		keep country id *`country'*
		rename v`country'* i*

		gegen i_rowsum=rowtotal(i1-i56)
		forvalues v=1/56 {
			gen c_down`v' = i`v'/i_rowsum
			replace c_down`v'=0 if c_down`v'==. 	// need zeros for matrix manipulation later on
		}
		forvalues v=1/56 {
			gegen i`v'_colsum=total(i`v')
			gen c_up`v' = i`v'/i`v'_colsum
			replace c_up`v'=0 if c_up`v'==. 		// need zeros for matrix manipulation later on
			drop i`v'_colsum
		}
		forvalues v=1/56 {
			gen c_down`v'_0D = c_down`v' 
			replace c_down`v'_0D = 0 if id==`v' 
		}
		forvalues v=1/56 {
			gen c_up`v'_0D = c_up`v' 
			replace c_up`v'_0D = 0 if id==`v' 
		}
		rename id bid
		drop i_rowsum i*
		gen year = `y'
		rename bid industry
		order country year industry
		compress
		save "$temp/COEF`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 2a: proxies of downIM, upEX, upIM, downEX with time-varying weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IM`country'_`y'_development_split.dta", clear
		merge 1:1 country industry year using "$temp/EX`country'_`y'_development_split.dta"
		drop _merge
		merge 1:1 country industry year using "$temp/COEF`country'_`y'.dta"
		drop _merge
		order country industry year 
		
		mkmat IM_dvlpd01-IM_dvlpd56, matrix(IM_dvlpd) 
		mkmat IM_dvlpng01-IM_dvlpng56, matrix(IM_dvlpng)
		mkmat EX1_dvlpd-EX56_dvlpd, matrix(EX_dvlpd) 
		mkmat EX1_dvlpng-EX56_dvlpng, matrix(EX_dvlpng)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in IM_dvlpd IM_dvlpng {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
		}
		foreach type in EX_dvlpd EX_dvlpng {
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
		}
		foreach var in down0dIM_dvlpng down0dIM_dvlpd downIM_dvlpng downIM_dvlpd ///
						up0dEX_dvlpng up0dEX_dvlpd upEX_dvlpng upEX_dvlpd {
			gen `var' = .
		}
		
		forvalues v=1/56 {
			foreach var in down0dIM_dvlpng down0dIM_dvlpd downIM_dvlpng downIM_dvlpd ///
							up0dEX_dvlpng up0dEX_dvlpd upEX_dvlpng upEX_dvlpd {
				replace `var' = `var'`v' if industry==`v'
				drop `var'`v' 
			}
		}
		keep country industry year down* up*
		save "$temp/IM`country'_`y'_TV.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_TV.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_TV.dta"
	}
	sort country industry year
	save "$temp/DOWNIMEXUP_`country'.dta", replace
}

use "$temp/DOWNIMEXUP_AUT.dta", clear
qui foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/DOWNIMEXUP_`country'.dta"
}
sort country industry year
save "$temp/DOWNIMEXUP_development_split.dta", replace

* Merge proxies
use "$temp/Proxies_development_split.dta", clear
merge 1:1 country industry year using "$temp/DOWNIMEXUP_development_split.dta"
drop _merge
order country industry year IM* EX* down0dIM* up0dIM* down0dEX* up0dEX*
compress
save "$temp/Proxies_development_split.dta", replace

* Housekeeping
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		cap erase "$temp/IM`country'_`y'.dta"
		cap erase "$temp/IM`country'_`y'_development_split.dta"
		cap erase "$temp/IM`country'_`y'_TV.dta"
		cap erase "$temp/IM`country'_`y'_00.dta"
		cap erase "$temp/COEF`country'_`y'.dta"
		cap erase "$temp/EX`country'_`y'.dta"
		cap erase "$temp/EX`country'_`y'_development_split.dta"
		cap erase "$temp/DOWNIMEXUP_development_split.dta"
		cap erase "$temp/DOWNIMEXUP_00.dta"
		cap erase "$temp/DOWNIMEXUP_00.dta"
		cap erase "$temp/DOWNIMEXUP_`country'.dta"
		cap erase "$temp/DOWNIMEXUP.dta"
		cap erase "$temp/Proxies_`country'.dta"
	}
}
